:- module(c_print, [print_optimized/3]).

/*********************** PRETTY PRINTER ***************************/
/* Declarations */
print_sym(_,[]).

print_sym(Stream,[H|T]):-
    H=..[Type,Name],
    write(Stream,Type),tab(Stream,1),write(Stream,Name),
    put_char(Stream,';'),nl(Stream),
    print_sym(Stream,T),!.

erase_sym([]).
erase_sym([H|T]):-
    erase(H),
    erase_sym(T).

print_symtable(Stream):-
	findall(X,recorded(symtable,X),B),
	print_sym(Stream,B),
	findall(X,recorded(symtable,_,X),B2),
	erase_sym(B2).

print_code(Stream,for(LB,UB,S,B),N):-
   tab(Stream,N),
   print(Stream,for),put_char(Stream,'('),
   print_expression(Stream,LB),
   print(Stream,';'),
   print_expression(Stream,UB),
   print(Stream,';'),
   print_expression(Stream,S),
   put_char(Stream,')'),tab(Stream,1),put_char(Stream,'{'),nl(Stream),
   print_code(Stream,B,N+2),
   tab(Stream,N),put_char(Stream,'}'),nl(Stream),flush_output(Stream),!.
print_code(Stream,if(C,P1,P2),N):-
   tab(Stream,N),
   print(Stream,if),
   print_expression(Stream,C),
   tab(Stream,1),put_char(Stream,'{'),nl(Stream),
   print_code(Stream,P1,N+2),
   tab(Stream,N),put_char(Stream,'}'),tab(Stream,1),print(Stream,else),
   put_char(Stream,'{'),nl(Stream),
   print_code(Stream,P2,N+2),
   tab(Stream,N),put_char(Stream,'}'),nl(Stream),flush_output(Stream),!.

print_code(Stream,if(C,P1),N):-
   tab(Stream,N),
   print(Stream,if),
   print_expression(Stream,C),
   tab(Stream,1),put_char(Stream,'{'),nl(Stream),
   print_code(Stream,P1,N+2),
   tab(Stream,N),put_char(Stream,'}'),nl(Stream),flush_output(Stream),!.

print_code(Stream,(B1;B2),N):-
   print_code(Stream,B1,N),print_code(Stream,B2,N),flush_output(Stream),!.

print_code(Stream,memcpy(A1,A2,S),N):-
   tab(Stream,N),
   print(Stream,memcpy),put_char(Stream,'('),
   print(Stream,A1),put_char(Stream,','),
   print(Stream,A2),put_char(Stream,','),
   print_expression(Stream,S),put_char(Stream,')'),
   put_char(Stream,';'),nl(Stream),flush_output(Stream),!.
print_code(Stream,memset(A1,A2,S),N):-
   tab(Stream,N),
   print(Stream,memset),put_char(Stream,'('),
   print(Stream,A1),put_char(Stream,','),
   print(Stream,A2),put_char(Stream,','),
   print(Stream,S),put_char(Stream,')'),
   put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

print_code(Stream,(A @= E),N):-
   tab(Stream,N),
   print_expression(Stream,A),
   print(Stream,=),
   print_expression(Stream,E),
   put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

print_code(Stream,(A #= E),N):-
   tab(Stream,N),
   print_expression(Stream,A),put_char(Stream,'%'),print(Stream,=),print_expression(Stream,E),put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

print_code(Stream,(A ~= E),N):-
   tab(Stream,N),
   print_expression(Stream,A),put_char(Stream,'|'),print(Stream,=),print_expression(Stream,E),put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

print_code(Stream,id(A,_),N):-
	tab(Stream,N),
	print(Stream,A),!.

print_code(Stream,E,N):-
	tab(Stream,N),print_expression(Stream,E),
	put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

/*
print_code(Stream,E,N):-
E=..[Op,A,E1],
   tab(Stream,N),
   print_expression(Stream,A),print(Stream,Op),print_expression(Stream,E1),put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

print_code(Stream,vector(Op,A,E1,E2),N):-
   tab(Stream,N),
   print(Stream,vector),put_char(Stream,'('),
   print(Stream,Op),put_char(Stream,','),
   print_expression(Stream,A),put_char(Stream,','),
   print_expression(Stream,E1),put_char(Stream,','),
   print_expression(Stream,E2),
   put_char(Stream,')'),put_char(Stream,';'),nl(Stream),flush_output(Stream),!.

*/

print_type(Stream,star(T)):-
	print_type(Stream,T),
	put_char(Stream,'*'),!.
print_type(Stream,struct(T)):-
	print(Stream,struct),
	put_char(Stream,' '),
	print(Stream,T),
	put_char(Stream,' '),!.
print_type(Stream,B):-
	print(Stream,B),put_char(Stream,' '),!.
print_decl(_,[],_).
print_decl(Stream,[id(A,T)|L],N):-
	tab(Stream,N),
	print_type(Stream,T),
	print(Stream,A),
	put_char(Stream,';'),nl(Stream),
	print_decl(Stream,L,N),!.
print_decl(_,[]).
print_decl(Stream,[id(A,T)|L]):-
	write(Stream,extern),
	put_char(Stream,' '),
	print_type(Stream,T),
	print(Stream,A),
	put_char(Stream,';'),nl(Stream),
	print_decl(Stream,L),!.

print_global(Stream,T):-
	analyze:workingset(T,L),
	print_decl(Stream,L),!.
print_local(Stream,T,N):-
	analyze:loopindex(T,L),
	print_decl(Stream,L,N),!.
print_array(Stream,[H|T]):-
   write(Stream,'['),
   print_expression(Stream,H),
   write(Stream,']'),
   print_array(Stream,T).

print_listdot(_,[]).
print_listdot(Stream,[H|T]):-
		   put_char(Stream,'.'),
		   print(Stream,H),
		   print_listdot(Stream,T).
print_listexpr(_,[]):-!.
print_listexpr(Stream,[H]):-
	print_expression(Stream,H),!.
print_listexpr(Stream,[H|T]):-
	print_expression(Stream,H),
	put_char(Stream,','),
	print_listexpr(Stream,T),!.

print_expression(Stream,call(A,E)):-
	write(Stream,A),
	put_char(Stream,'('),
	print_listexpr(Stream,E),
	put_char(Stream,')'),!.	
print_expression(Stream,star(E)):-
    put_char(Stream,'('),
    put_char(Stream,'*'),
    print_expression(Stream,E),
   put_char(Stream,')'),!.
print_expression(Stream,(A @= E)):-
   print_expression(Stream,A),
   print(Stream,=),
   print_expression(Stream,E),!.
print_expression(Stream,[id(A,star(_))|E]):-
   print(Stream,A),
   write(Stream,'['),
   print_expression(Stream,E),
   write(Stream,']'),!.
print_expression(Stream,[A|E]):-
   print_expression(Stream,A),
   write(Stream,'['),
   print_expression(Stream,E),
   write(Stream,']'),!.
print_expression(Stream,id(A,_)):-
   print(Stream,A),!.
print_expression(Stream,E):-
   E=..[Op,A1,A2],
   put_char(Stream,'('),
   print_expression(Stream,A1),
   print(Stream,Op),
   print_expression(Stream,A2),put_char(Stream,')'),!.
print_expression(Stream,incr(A)):-
   print_expression(Stream,A),
   put_char(Stream,'+'),
   put_char(Stream,'+'),!.
print_expression(Stream,E):-
   E=..[Op,A],
   print(Stream,Op),
   print_expression(Stream,A),!.
print_expression(Stream,E):-print(Stream,E).

print_includes(_,[]).
print_includes(Stream,[ginc(I)|T]):-
	put_char(Stream,'#'),
	write(Stream,include),
	put_char(Stream,' '),
	put_char(Stream,'<'),
	write(Stream,I),
	put_char(Stream,'.'),
	put_char(Stream,'h'),	
	put_char(Stream,'>'),
	nl(Stream),print_includes(Stream,T),
	!.
print_includes(Stream,[inc(I)|T]):-
	put_char(Stream,'#'),
	write(Stream,include),
	put_char(Stream,' '),
	put_char(Stream,'\"'),
	write(Stream,I),
	put_char(Stream,'.'),
	put_char(Stream,'h'),	
	put_char(Stream,'\"'),
	nl(Stream),print_includes(Stream,T),!.

concat_parameter_value(N1,H,N2):-
  number_chars(H,C),atom_chars(A,C),concat_atom([N1,A],'_',N2).

print_optimized_program(Stream,H):-
  new_param(X),
  /* Name of the function */
  concat_parameter_value(kernel,X,KernelName),
  write(Stream,void),put_char(Stream,' '),
  write(Stream,KernelName),put_char(Stream,'('),put_char(Stream,')'),
  put_char(Stream,'{'),nl(Stream),
  print_local(Stream,H,2),
  print_code(Stream,H,2),	       
  put_char(Stream,'}'),nl(Stream).

print_optimized(File,Includes,[H|L]):-
  /* Name of the file */
  concat_atom([File,kernel],'_',ZZ),
  concat_atom([ZZ,c],'.',Z),
  open(Z,write,Stream),
  print_includes(Stream,Includes),
  print_global(Stream,H),
  maplist(print_optimized_program(Stream),[H|L]),
  close(Stream).

print_optimized(_,_,[]).

new_param(X):-flag(param,Old,Old+1),X is Old.
reset_param:-flag(param,_,0).

