/*---------------------------------------------------------------------
 * gene.c
 * Generation of two isomorphic digraphs with an assigned number
 * of nodes and branches.
 * The resulting graphs are always connected; the number
 * of branches may be greater of the one specified by the user
 * in order to enforce this property.
 * The graphs are stored as adjacency matrices into text files
 * with the format:
 *         no_of_nodes
 *         a11 a12 a13 ...  a1n
 *         a21 a22 ...      a2n
 *         an1 an2 ...      ann
 * where aij is 1 if there is a branch from node i to node j, 
 * 0 otherwise.
 ---------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <assert.h>

#define MAX 1000

#ifndef RAND_MAX
#define RAND_MAX 0x7FFFFFFF
#endif

typedef unsigned char byte;

static byte mat[MAX][MAX];
static short s[MAX];

void main(int argc, char *argv[])
  { int nodes, branches; 
    int i, j, n1, n2;
    FILE *f1, *f2;

    if (argc!=5)
      { fprintf(stderr, "Usage: gene nodes branches file1 file2\n");
        return;
      }
    nodes=atoi(argv[1]);
    assert(nodes>0 && nodes<=MAX);
    branches=atoi(argv[2]);
    assert(branches>=0 && branches<=nodes*(nodes-1));
    f1=fopen(argv[3], "w");
    if (!f1)
      { fprintf(stderr, "Can't open file: %s\n", argv[3]);
        return;
      }
    f2=fopen(argv[4], "w");
    if (!f2)
      { fprintf(stderr, "Can't open file: %s\n", argv[4]);
        return;
      }


        /* Generate the matrix */
    srand(time(NULL));
    for(i=0; i<branches; i++)
          { do {
            n1=floor(rand()/(float)RAND_MAX*nodes);
            n2=floor(rand()/(float)RAND_MAX*nodes);
            } while (n1==n2 || mat[n1][n2]==1);
            mat[n1][n2]=1;
          }

        /* Ensure the graph is connected */
    for(i=1; i<nodes; i++)
          { for(j=0; j<i; j++)
              if (mat[i][j] || mat[j][i])
                break;
            if (i==j)
              { j=floor(rand()/(float)RAND_MAX*i);
                if (rand()<0.5) mat[i][j]=1;
                else mat[j][i]=1;
              }
          }
                

    fprintf(f1, "%d\n", nodes);
    for(i=0; i<nodes; i++)
          { for(j=0; j<nodes; j++)
              fprintf(f1, "%d ", mat[i][j]);
            fprintf(f1, "\n");
          }

        /* permutation of the matrix */
     for(i=0; i<nodes; i++)
          s[i]=i;
     for(i=0; i<nodes; i++)
          { int tmp;
            j=floor(rand()/(float)RAND_MAX*nodes);
            tmp=s[j];
            s[j]=s[i];
            s[i]=tmp;
          }
            
    fprintf(f2, "%d\n", nodes);
    for(i=0; i<nodes; i++)
          { for(j=0; j<nodes; j++)
              fprintf(f2, "%d ", mat[s[i]][s[j]]);
            fprintf(f2, "\n");
          }

     fclose(f1);        
     fclose(f2);        
  }

