/*-------------------------------------------------------------------
 * graph.cc
 * Implementation of the Graph class 
 ------------------------------------------------------------------*/

#include "graph.h"

#include <stdio.h>
#include "error.h"

/*----------------------------------------------------------------
 * methods of the class Graph
 ---------------------------------------------------------------*/

/*------------------------------------------------------------------
 * Graph::Graph(filename)
 * Constructor.
 * Loads a graph from a file. The format of the file is
 *       n
 *       a11 a12 ... a1n
 *       a21 a22 ... a2n
 *       ...
 *       an1 an2 ... ann
 * where n is the number of nodes, aij is non-zero if there is an
 * edge from node i to node j.
 *-----------------------------------------------------------------*/
Graph::Graph(char *filename)
  { FILE *f=fopen(filename, "r");
    if (!f)
      error("can't open file: %s", filename);
    if (fscanf(f, "%d", &n)!=1)
      error("Can't read node number in file: %s", filename);

    matrix=new byte *[n];
    out=new node_id *[n];
    in=new node_id *[n];
    in_count=new int[n];
    out_count=new int[n];
    
    if (!matrix || !in || !out || !in_count || !out_count)
      error("Out of memory");

    int i;
    for(i=0; i<n; i++)
      in_count[i]=out_count[i]=0;

    for(i=0; i<n; i++)
      { matrix[i]=new byte[n];
        if (!matrix[i])
          error("Out of memory");

        int j, datum;
        for(j=0; j<n; j++)
          { if (fscanf(f, "%d", &datum)!=1)
              { char buf[128];
                sprintf(buf, "Can't read item (%d, %d) from file: %%s", i, j);
                error(buf, filename);
              }
            if (datum)
              { matrix[i][j]=1; 
                out_count[i]++;
                in_count[j]++;
              }
            else
              matrix[i][j]=0;
          }
      }

    fclose(f);
    
    for(i=0; i<n; i++)
      { out[i]=new node_id[out_count[i]];
        if (!out[i])
          error("Out of memory"); 
        int j,k;
        for(k=0, j=0; j<n; j++)
          if (matrix[i][j])
            out[i][k++]=j;
      }
    for(i=0; i<n; i++)
      { in[i]=new node_id[in_count[i]];
        if (!in[i])
          error("Out of memory"); 
        int j,k;
        for(k=0, j=0; j<n; j++)
          if (matrix[j][i])
            in[i][k++]=j;
      }

  }

/*------------------------------------------------------------------
 * Graph::Graph(f)
 * Constructor.
 * Loads a graph from a file. The format of the file is
 * described in the comment of Graph::Graph(char *filename).
 * Notice that this constructor does not close the file,
 * allowing several graphs to be read from one file. 
 *-----------------------------------------------------------------*/
Graph::Graph(FILE *f)
  { 
    if (fscanf(f, "%d", &n)!=1)
      error("Can't read node number in file");

    matrix=new byte *[n];
    out=new node_id *[n];
    in=new node_id *[n];
    in_count=new int[n];
    out_count=new int[n];
    if (!matrix || !in || !out || !in_count || !out_count)
      error("Out of memory");

    int i;
    for(i=0; i<n; i++)
      in_count[i]=out_count[i]=0;

    for(i=0; i<n; i++)
      { matrix[i]=new byte[n];
        if (!matrix[i])
          error("Out of memory");

        int j, datum;
        for(j=0; j<n; j++)
          { if (fscanf(f, "%d", &datum)!=1)
              { char buf[128];
                sprintf(buf, "Can't read item (%d, %d) from file", i, j);
                error(buf);
              }
            if (datum)
              { matrix[i][j]=1; 
                out_count[i]++;
                in_count[j]++;
              }
            else
              matrix[i][j]=0;
          }
      }

    for(i=0; i<n; i++)
      { out[i]=new node_id[out_count[i]];
        if (!out[i])
          error("Out of memory"); 
        int j,k;
        for(k=0, j=0; j<n; j++)
          if (matrix[i][j])
            out[i][k++]=j;
      }
    for(i=0; i<n; i++)
      { in[i]=new node_id[in_count[i]];
        if (!in[i])
          error("Out of memory"); 
        int j,k;
        for(k=0, j=0; j<n; j++)
          if (matrix[j][i])
            in[i][k++]=j;
      }

  }


/*-------------------------------------------------
 * Graph::~Graph()
 * Destructor.
 * Frees the memory allocated for the graph
 ------------------------------------------------*/
Graph::~Graph()
  { int i;
    for(i=0; i<n; i++)
      { delete[] matrix[i];
        delete[] in[i];
        delete[] out[i];
      }
    delete[] matrix;
    delete[] in_count;
    delete[] in;
    delete[] out_count;
    delete[] out;
  }


/*-------------------------------------------------------------------
 * void Graph::VisitInEdges(node, vis, param)
 * Applies the visitor to all the 'in' edges of 'node'
 ------------------------------------------------------------------*/
void Graph::VisitInEdges(node_id node, edge_visitor vis, param_type param)
  { 
    assert(node<n);
    int i;
    for(i=0; i<in_count[node]; i++)
      vis(this, in[node][i], node, param);
  }

/*-------------------------------------------------------------------
 * void Graph::VisitOutEdges(node, vis, param)
 * Applies the visitor to all the 'out' edges of 'node'
 ------------------------------------------------------------------*/
void Graph::VisitOutEdges(node_id node, edge_visitor vis, param_type param)
  {
    assert(node<n); 
    int i;
    for(i=0; i<out_count[node]; i++)
      vis(this, node, out[node][i], param);
  }

/*-------------------------------------------------------------------
 * void Graph::VisitEdges(node, vis, param)
 * Applies the visitor to all the edges of 'node'
 ------------------------------------------------------------------*/
void Graph::VisitEdges(node_id node, edge_visitor vis, param_type param)
  { VisitInEdges(node, vis, param);
    VisitOutEdges(node, vis, param);
  }

