/*
 * automorphisms_filter.c: keep non-isomorphic subsets A 
 *
 * author: Arnaud Pcher <pecher@lifo.univ-orleans.fr>
 *
 * this program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 */

#include <stdio.h>

#define nbmax 200 /* maximum order of group G */ 
#define false 0 
#define true 1 
#define sizeautmax 25000 /* maximum order of automorphisms group AUT(G) */
#define outputmax 100000 /* maximum number of subsets A after automorphism filter */
#define omegamax 15 /* maximum size of subet A */

typedef int set[omegamax+1]; /* subset A ranges from A[1] to A[omega] */
typedef int map[nbmax+1]; /* for one automorphism */
typedef map listaut[sizeautmax]; /* for the list of automorphisms (AUT(G) */
typedef set list[outputmax]; /* for the list of subsets after automorphism filter */
typedef int boolean;

int nbautomor,n,omega;
set A;
boolean stop;
listaut aut;
list input;

boolean empty(set A)
     /* true iff A is the empty set */
{
  int i=1; boolean b=true;
  do {
    b = (b && (A[i]==0));
    i++;
  } while ((b) && (i<=omega));
  return (b);
}  /* empty */

void printset(set A)
     /* print a set A on a line */
{
  int i;
  for (i=1; i<=omega; i++) printf("%4d",A[i]);
  printf("\n");
}

boolean equalset(set A, set B)
     /* true iff A=B (A and B are not necessarily ordered) */
{
  boolean b=true; int i=1, j=1;
  for (i=1; ((i<=omega)&&(b==true)); i++)
    {
		b=false;
		for (j=1; j<=omega; j++)
			{if (A[i]==B[j]) b=true;}
    };			
  return b;
}

boolean less_or_equal(set A, set B)
     /* true iff A<=B in lexicographic ordering */
{
  int i=1; 
  for (i=1; ((i<=omega) && (A[i]==B[i])); i++);
  if (i>omega) {return true;}
  else return(A[i]<=B[i]);		 
}

boolean find_dicho(set A, list input, int start, int stop)
     /* true iff A is in list input (algorithm: dichotomic search) */
{
  int middle;
  
  middle=((start+stop)/2);
  if (stop<=start) 
    {return(equalset(A,input[start]));}
  else
    {
      if (less_or_equal(A,input[middle])==true) 
	{return(find_dicho(A,input,start,middle));}
      else
	{return(find_dicho(A,input,(middle+1),stop));};
    };
}

void insertion_sort(set A)
     /* sort A (algorithm: insertion) */
{int i,j,key;

 for (j=2; j<=omega; j++)
   {
     key=A[j];
     i=j-1;
     for (i=i; ((i>0) && (A[i]>key)); i--) A[i+1]=A[i];
     A[i+1]=key;
     ;}
}

boolean inlist(set A, list input, int k)
     /* true iff there exists an automorphism f, such that f(A) is one of input[1], ..., input[k] */
{boolean b=false; int j,d; set image;

 for (j=1; ((j<=nbautomor) && (b==false)); j++) /* we go through the whole list of automorphisms */
   {
     /* calculate the image of A under aut[j] */
     for (d=1; d<=omega; d++) image[d]=aut[j][A[d]];
     /* sort this image before applying dichotomic search */
     insertion_sort(image);
     if (find_dicho(image,input,1,k)==true) b=true;
   }
 return b;
}

int main (void)
{
  long int i,line;
  int j,col,data;
  FILE *error;

  scanf("%d%*[^\n]",&n); /* read n (order of the group G)*/
  scanf("%d%*[^\n]",&omega); /* read omega (size of A)*/
  if (omega>omegamax)
    {   
      /* write final line marker */
      for (col=1; col<=omega; col++) {printf("%4d",0);}
      printf("\n");
      error=fopen("./tmp/error","w");
      fprintf(error,"--- automorphisms_filter ---\n");
      fprintf(error,"Overload: omega is too big (max value is set to %d) \n",omegamax);
      fprintf(error,"Exiting");
      fclose(error);
      return(1);
    }
  /* read automorphisms of the group G */
  scanf("%d%*[^\n]",&nbautomor);
  if (nbautomor>=sizeautmax)
    {   
      /* write final line marker */
      for (col=1; col<=omega; col++) {printf("%4d",0);}
      printf("\n");
      error=fopen("./tmp/error","w");
      fprintf(error,"--- automorphisms_filter ---\n");
      fprintf(error,"Overload: order of automorphism group is too big (max value is set to %d) \n",sizeautmax);
      fprintf(error,"going on anyway, with only the first %d automorphisms of AUT(G) \n",sizeautmax-1);
      fclose(error);
      nbautomor=sizeautmax-1;
    }
  for (i=1; i<=nbautomor; i++)
    {
      /* read automorphism number i */
      for (j=1; j<=n; j++)
	{scanf("%d",&data); aut[i][j]=data;}
      scanf("\n");
    }
  /* process the list of the candidates A */
  /* end if stop=true, that is if A is empty */
  line=1; /* intialisation of counter line */
  stop=false;
  do
    {
      for (col=1; col<=omega; col++) /* read A */
	{scanf("%d",&data); A[col]=data;}
      scanf("/n"); /* go to next line */
      if (empty(A)==true) {stop=true;} 
      else
	/* look for an image of A (under AUT(G)) in input[1],...,input[line-1] */
	if (inlist(A,input,(line-1))==false)  
	  {
	    printset(A); 
	    /* if not store A in input[line] */
	    for (i=1; i<=omega; i++) input[line][i]=A[i];
	    line++;
	    if (line>=outputmax) 
	      { 
		/* write final line marker */
		for (col=1; col<=omega; col++) {printf("%4d",0);}
  		printf("\n");
	      	error=fopen("./tmp/error","w");
		fprintf(error,"--- automorphisms_filter ---\n");
		fprintf(error,"Overload: list of candidates A AFTER automorphism filter is too big (max value is set to %d) \n",outputmax);
		fprintf(error,"Exiting \n");
		fclose(error);
		return(1);
	      };
	  };
    } while (stop==false);
  /* write final line marker */
  for (col=1; col<=omega; col++) {printf("%4d",0);}
  printf("\n");
  return 0;
}
