/*
 * calculate_B.c: looks for a subset B such that (A,B) 
 *                is a near-factorization 
 *                (if so: prints A and B on standard output) 
 *         input: list of candidates A 
 *        output: list of A with B such that 
 *                (A,B) is a near-factorization
 *            
 *                ! do not give all B such that (A,B) is a near-factorization
 *                (but at least one if there is any)
 *
 * author: Arnaud Pcher <pecher@lifo.univ-orleans.fr>
 *
 * this program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 *
 */

#include <stdio.h>

#define nbmax 200 /* maximum order of group G */
#define omegamax 50 /* maximum size of A (and of B) */
#define false 0
#define true 1

typedef int boolean;
typedef boolean group[nbmax + 1]; /* to store a subset of G */ 
typedef struct set{
  int element[omegamax+1];
  int size;
} set; /* to store A and B */
typedef int permutation[nbmax + 1]; /* to store inverting map */
typedef int matrix[nbmax+1][nbmax+1]; /* to store Cayley table of the group G */

group AB; /* to store A*B */
set  A, B, C;
int numero, omega, alpha, data, i, j, n, IdentityElement;
boolean stop;
matrix M;
permutation inv; /* inverting map */
boolean force_symmetric_flag; /* if true, restrict attention to symmetric subsets only */
boolean found;

void printset(set A)
     /* print the set A on a line */
{
  int i;

  for (i = 1; i<=A.size; i++) printf("%4d ",A.element[i]);
}  

void load_matrix(matrix M)
     /* store Cayley table of groupe G in matrix M */
     /* load data from standard input */
{
  int i,j;

  for (i=1; i<=n; i++) 
    {
      for (j=1; j<=n; j++) 
	scanf("%d", &M[i][j]);
      scanf("\n"); /* go to next line */
    }
}

int identity(matrix M, int n)
     /* returns identity element in group G */
{
  boolean b;
  int i, j, temp=0;

  for (j = 1; j <= n; j++) {
    b = true;
    for (i = 1; i <= n; i++)
      b = (b && (M[i][j]==i));
    if (b)
      temp = j;
  }
  return temp;
}

int inverse(int i)
     /* returns inverse of element i */
{
  int j = 0; 

  do {
    j++;
  } while (M[j][i] != IdentityElement);
  return j;
}

void set_empty_set(set *A)
     /* A:=empty set */
{
  A->size=0;
}


int multiplication(int e, int f)
     /* returns e*f */
{
  return(M[e][f]);
}

void put_in_set(int e, set *A)
     /* A:=A union {e} */
{
  A->size++;
  A->element[A->size]=e;
}

void remove_from_set(int e, set *A)
     /* A:= A\{e} */
{
  int i,j;
  for (i=1; ((A->element[i]!=e) && (i<=A->size)); i++); /* find position of element e */
  for (j=i; j<=A->size-1; j++) A->element[j]=A->element[j+1]; /* remove e and copy the end */
  A->size--; /* adjust size */
}
  
void multiplication_set_element(set A, int e, set *B)
     /* B:= Ae */
{
  int i;

  set_empty_set(B);
  for (i = 1; i<=A.size; i++) 
    put_in_set(multiplication(A.element[i],e),B);
}  

boolean empty(set A)
     /* true iff A=empty set */
{
  return(A.size==0);
}

boolean is_nf(group AB) 
     /* true iff AB minus one element is filled */
     /* that is iff A*B is a near-factorization */
{
  int counter=0;
  int i=1;

  for (i=1; i<=n; i++)
    if (AB[i]==false)
      {
	counter++;
	if (counter>=2) return false; /* there are at least two uncovered elements */
      }
  return true;
}

boolean room(group AB, set A, int e)
     /* true iff (Ae intersection AB)=empty set */
{
  int i;
  boolean b;
  set F;
  
  multiplication_set_element(A, e, &F);
  b = true;
  i = 1;
  do 
    {
      if (AB[F.element[i]]==true)
	b=false; /* conflict */
      i++;
    } 
  while (b && (i<=F.size));
  return(b);
}

void put(group AB, set A, int e)
     /* AB:=(AB union Ae) */
{
  int i;
  set F;

  multiplication_set_element(A, e, &F);
  for (i=1; i<=F.size; i++)
     AB[F.element[i]]=true;     
}

void remove_subset(group AB, set A, int e)
     /* AB:=AB\Ae */
{
  int i;
  set F;

  multiplication_set_element(A, e, &F);
  i = 1;
  for (i=1; i<=F.size; i++)
    AB[F.element[i]]=false;
}

void calculate_symmetric_B(group AB, set A, set B)
     /* find and print symmetric subsets B (if any) such that (A,B) is a near-factorization */
{
  int i,j,x;

  if (found) return;

  /* print solutions */
  if (is_nf(AB)) 
    {
      found=true;
      printset(A);
      printf("-- B=");
      printset(B);
      printf(" -- \n");
      return;
    }
  for (i=1; i <=n; i++)
    if (i<=inv[i]) // since A and B are kept symmetric, if i is covered then inv[i] also, thus it is enough to consider the cover of the smallest
      if (AB[i]==false)
	{
	  for (j=1; j<=omega; j++)
	    {
	      /* make recursive call */
	      x=multiplication(inv[A.element[j]],i); /* x=a^-1*i */
	      if (x<=inv[x]) // x : to be likely stored in B (thus inv[x] also) 
		if (room(AB,A,x)==true)
		  {
		    put(AB,A,x);
		    put_in_set(x,&B);
		    if (x==inv[x])
		      {
			calculate_symmetric_B(AB,A,B);
		      }
		    else
		      if (room(AB,A,inv[x])==true)
			{
			  put(AB,A,inv[x]);
			  put_in_set(inv[x],&B);
			  calculate_symmetric_B(AB,A,B);
			  remove_from_set(inv[x],&B);
			  remove_subset(AB,A,inv[x]);
			}
		    remove_from_set(x,&B);
		    remove_subset(AB,A,x);
		  }
	    }
	}
}

void CalculateB(group AB, set A, set B)
/* find and print subsets B (if any) such that (A,B) is a near-factorization */
{
  int i,j,x;
  int nbchoice, nbchoice_min, nbchoice_max, i_min; 
    
  if (found) return;

  /* print solutions */
  if (is_nf(AB)) 
    {
      found=true;
      printset(A);
      printf("-- B=");
      printset(B);
      printf(" -- \n");
      return;
    }
  /* find pivot (nbchoice_min and i_min) */
  nbchoice_min=omega; 
  i_min=1; 
  nbchoice_max=0;
  for (i=1; i <=n; i++) 
  	if (AB[i]==false)
  		{
		  nbchoice=0;
		  for (j=1; j<=omega; j++)
		    {
		      x=multiplication(inv[A.element[j]],i); /* x=a^-1*i */
		      if (room(AB,A,x)==true) nbchoice++;
		    }
		  if (nbchoice>=nbchoice_max) nbchoice_max=nbchoice;
		  if (nbchoice<=nbchoice_min)
		    if (nbchoice>0)
		      {
			nbchoice_min=nbchoice;
			i_min=i;
		      }
		}
  /* make recursive call */
  if (nbchoice_max>0) /* if not, nothing can be done */
    {
      for (j=1; j<=omega; j++)
	{
	  x=multiplication(inv[A.element[j]],i_min); /* x=a^-1*i */
	  if (room(AB,A,x)==true) 
	    {
	      put(AB,A,x);
	      put_in_set(x,&B);
	      CalculateB(AB,A,B);
	      remove_from_set(x,&B);
	      remove_subset(AB,A,x);
	    }
	}
    }
}

int main(void)
{ 
  /* load data */  
  scanf("%d%*[^\n]",&data); /* see if restriction to symmetric subsets applies */
  force_symmetric_flag=(data==1);
  scanf("%d%*[^\n]",&n); /* read n, the order of the group */
  scanf("%d%*[^\n]",&omega); /* read omega, the size of A */
  alpha=((n-1) / omega); /* calculate alpha, the size of B (n=alpha*omega+1) */ 
  load_matrix(M); /* load Cayley table of group */
  IdentityElement=identity(M,n); /* determine identity element */
  for (i=1; i<=n; i++) inv[i]=inverse(i); /* calculate inverting map */
  /* input file processing */
  do 
    {
      set_empty_set(&A);
      set_empty_set(&B);
      for (i = 1; i <=n; i++) AB[i]=false; /* at first A*B=empty set */
      for (j = 1; j <=omega; j++) /* load A */
	{
	  scanf("%d",&data);
	  if (data!=0) put_in_set(data,&A); 
       }
     scanf("\n");
     stop = empty(A);
     if (!stop) 
       {
	 found=false;
	 //	 if (force_symmetric_flag) 
	 //  {calculate_symmetric_B(AB,A,B);}
	 // else
	   CalculateB(AB,A,B);
       };
   } while (!stop); 
 return 0;
}
/* End. */





