/*
 * calculate_associated_graph.c: calculates associated graph to 
 *                               a given near-factorization and 
 *                               compute some values of parameters  *                               of this graph 
 *                        input: n, omega, a near-factorization 
 *                               given by a line of syntax 
 *                               0 1 2 -- B= 0 3 6 9 
 *                               (A={0,1,2} and B={0,3,6,9})
 *                       output: adjency matrix of graph G(A,B) 
 *                               INT(A), INT(B), 
 *                               INT(A^-1), INT(B^-1), 
 *                               list of the vertices 
 *                               that A is a mother of, 
 *                               list of vertices that 
 *                               B^-1 is a father of 
 *
 * author: Arnaud Pcher <pecher@lifo.univ-orleans.fr>
 *
 * this program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 *
 */

#include <stdio.h>

#define nbmax 200 /* maximum order of group G */
#define false 0
#define true 1

typedef int boolean;
typedef boolean set[nbmax + 1]; /* to store ConnectingSet, A and B */
typedef int permutation[nbmax + 1]; /* to store inverting map */
typedef int matrix[nbmax+1][nbmax+1]; /* to store Cayley table of group G */

set A, B, ConnectingSet;
int omega, alpha, data, i, j, n, IdentityElement;
boolean stop;
matrix M; /* Cayley table of G */
permutation inv; /* inverting map */

void PrintSet(set A)
     /* print A on a line */
{
  int i;
  
  for (i = 1; i<=n; i++) {
    if (A[i])
      printf("%4d ", i);
  }
  printf("\n");
}

void LoadMatrix(matrix M)
     /* load Cayley table of group in M from standard input */
{
  int i,j;

  for (j=1; j<=n; j++) 
    {
      for (i=1; i<=n; i++) 
	scanf("%d", &M[i][j]);
      scanf("\n"); /* go to next line */
    }
}

int Identity(matrix M, int n)
     /* returns identity element in G */
{
  boolean b;
  int i, j, temp=0;

  for (j = 1; j <= n; j++) {
    b = true;
    for (i = 1; i <= n; i++)
      b = (b && (M[i][j]==i));
    if (b)
      temp = j;
  }
  return temp;
}

int Inverse(int i)
     /* returns inverse element of i */
{
  int j = 0; 

  do {
    j++;
  } while (M[j][i] != IdentityElement);
  return j;
}  

void SetEmptySet(set A)
{
  int i;

  for (i = 1; i <=n; i++) A[i] = false;
}  

int Multiplication(int e, int f)
     /* returns e*f */
{
  return M[e][f];
}

void MultiplicationElementSet(int e, set A, set B)
     /* B:=eA */
{
  int i;

  SetEmptySet(B);
  for (i = 1; i<=n; i++) 
    if (A[i]) 
      B[Multiplication(e,i)]=true;
}  

int Cardinal(set A)
     /* returns number of elements of A */
{
  int i,compt=0;

  for (i=1; i<=n; i++)
    if (A[i]) compt++;
  return compt;
}

void Intersection(set A, set B, set C)
     /* C:= A intersection B */
{
  int i;

  SetEmptySet(C);
  for (i=1; i<=n; i++)
    if (A[i])
      if (B[i]) 
	C[i]=true;
}

int INTER(set A)
     /* returns INT(A) */
{
  int i,max=0; set B,C;

  for (i=1; i<=n; i++)
    if (!(i==IdentityElement)) /* avoid set A */
      {
	MultiplicationElementSet(i,A,B);
	Intersection(A,B,C); /* C:=A intersection iA */
	if (Cardinal(C)>max) max=Cardinal(C);
      };
  return max;
}

boolean IsMotherOf(set A, int x)
     /* true iff A is a mother of x */
{
  int i; boolean b=false; set B,C;

  if (A[x]) b=true;
  for (i=1; i<=n; i++)
    {
      MultiplicationElementSet(i,A,B);
      if (B[x]) 
	{
	  Intersection(A,B,C);
	  if (Cardinal(C)<=1) b=false;
	};
    };
  return b;
}

void CalculateConnectingSet(set A, set B)
     /* B:=(A^-1*A)\{e} */
{
  int i,j,k;
 
  SetEmptySet(B);
  for (i=1; i<=n; i++)
    if (A[i]==true)
      {
	for (j=1; j<=n; j++)
	  {
	    if (A[j]==true)
	      {
		k=Multiplication(inv[i],j);
		B[k]=true;
	      };
	  };	
      };	
  B[IdentityElement]=false;
}	

void PrintGraph(set A)
     /* input A : connecting set of the graph */
     /* print n, print adjency matrix of the graph */
{
  int i,j,k;

  printf("%d\n",n);
  for (i=1; i<=n; i++)
    {
      for (j=1; j<=n; j++)
	{
	  k=Multiplication(inv[i],j);
	  if (A[k]==true) 
	    {printf("1 ");}
	  else 
	    printf("0 ");
	};
      printf("\n");
    };	
}

void PrintStats(set A, set B)
     /* input: clique maximum A, set B such that (A,B) is a near-factorization */
     /* print number of maximum cliques distinct of A meeting A; print parameters INT(A), INT(B), INT(inv(A)), INT(inv(B)); print list of vertices A is a mother of, print list of vertices inv(B) is a father of */
{
  set ConnectingSet,invA,invB; 

  SetEmptySet(invA);
  SetEmptySet(invB);
  for (i=1; i<=n; i++)
    if (A[i]) invA[inv[i]]=true; /* invA:=A^-1 */
  for (i=1; i<=n; i++)
    if (B[i]) invB[inv[i]]=true; /* invB:=B^-1 */
  CalculateConnectingSet(A,ConnectingSet);
  
  printf("Number of maximum cliques distinct of A intersecting A = %d\n", Cardinal(ConnectingSet));
  printf("INT(A)=%d\n",INTER(A));
  printf("INT(B)=%d\n",INTER(B));
  printf("INT(A^-1)=%d\n",INTER(invA));
  printf("INT(B^-1)=%d\n",INTER(invB));
  printf("A is a mother of :");
  for (i=1; i<=n; i++)
    if (A[i]) 
      if (IsMotherOf(A,i))
	printf("%d ", i);
  printf("\n");
  printf("B^-1 is a father of :");
  for (i=1; i<=n; i++)
    if (invB[i]) 
      if (IsMotherOf(invB,i))
	printf("%d ", i);		
  printf("\n");
}	

int main(void)
{
  /* load datas */  
  scanf("%d%*[^\n]",&n); /* read n, the number of vertices*/
  scanf("%d%*[^\n]",&omega); /* read omega, the size of A */
  alpha=(n-1)/omega; /* n=alpha*omega+1 */
  SetEmptySet(A);
  SetEmptySet(B);
  /* load A */
  for (j = 1; j <=omega; j++) {
    scanf("%d", &data);
    A[data] = true;
  };
  /* load B */
  scanf(" -- B=");
  for (j=1; j<=alpha; j++) {
  	scanf("%d", &data);
	B[data]=true;
	};
  scanf("%*[^\n]");
  
  LoadMatrix(M); /* load Cayley table M of the group */
  IdentityElement=Identity(M,n);
  for (i=1; i<=n; i++) inv[i]=Inverse(i); /* initialisation of inverting map */
  CalculateConnectingSet(A,ConnectingSet); /* initialisation of ConnectingSet */
  PrintGraph(ConnectingSet); /* print adjency matrix of graph */
  PrintStats(A,B); /* print some parameters of the graph */
 
  return 0;
}






