/*
 * generate_input_file.c: produces list of subsets containing 
 *                        identity element and, 
 *                        satisfying some properties (see below), 
 *                        of size omega of a given group 
 *                        with elements 1,...,n 
 *
 *                        recursive construction: ouput list 
 *                        is ordered through lexicographic order
 *                         
 *                        ! to lower the number of cases, A always contains the identity element, in the case NON-symetric  
 *
 *                        Optional:
 *                         - cut of search space
 *                         - Equipartition filter 
 *                         - Dispersion filter 
 *                         - Intersection filter 
 *                         - Restriction to symetric subsets
 *
 * author: Arnaud Pcher <pecher@lifo.univ-orleans.fr>
 *
 * this program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#define nbmax 200 /* maximal order of the group */
#define kmax 500 /* maximal number of subgroups for equipartition and dispersion filters  */
#define false 0
#define true 1

typedef int boolean;
typedef boolean Set[nbmax+1]; /* to store a subSet A */
typedef int Permutation[nbmax+1]; /* to store inverting map */
typedef int Matrix[nbmax+1][nbmax+1]; /* to store Cayley table of group G */
typedef struct Group{
  Set elements;
  int order;
} Group;
typedef Group SubGroupsList[kmax+1]; /* to store a list of subgroups */
typedef struct Partition{
  int split[nbmax+1]; /* to store a partition in right cosets */
  int index; /* number of right cosets */
} Partition;

Set A;
int i,j,k,data,n,number,omega;
int IdentityElement;
Matrix M; /* Cayley table */
Permutation inv; /* inverting map */
int CutFactor;
int NumberOfEntry=0;

int NumberOfSubgroupsForEquipartition, NumberOfSubgroups, NumberOfNormalSubgroups;
SubGroupsList SubgroupForEquipartition, Subgroup, NormalSubgroup;
Partition EquipartitionPartition[kmax+1]; /* EquipartitionPartition[i] stores the partition in right cosets of SubgroupForEquipartition[i] */
Partition SubgroupPartition[kmax+1]; /* SubgroupPartition[i] stores the partition in right cosets of Subgroup[i] */
Partition NormalSubgroupPartition[kmax+1]; /* NormalSubgroupPartition[i] stores the partition of right cosets of NormalSubgroup[i] */

boolean RandomizedFlag; /* if true, randomized cut of search space applies */
boolean EquipartitionFlag; /* if true, equipartition filter applies */
boolean DispersionFlag; /* if true, dispersion filter applies */
boolean IntersectionFlag; /* if true, intersection filter applies */
boolean ForceSymmetricFlag; /* if true, restrict attention to symmetric subsets only */

void SetEmptySet(Set A)
     /* A:= empty set */
{
  int i;

  for (i = 1; i <=n; i++) A[i] = false;
}  

void LoadMatrix(Matrix M)
     /* load Cayley table of group in M from standard input */
{
  int i,j;

  for (j=1; j<=n; j++) 
    {
      for (i=1; i<=n; i++) 
	scanf("%d", &M[i][j]);
      scanf("\n"); /* passage a la ligne suivante */
    }
}  

int Identity(Matrix M, int n)
     /* returns identity element in G */
{
  boolean b;
  int i, j, temp=0;

  for (j = 1; j <= n; j++) {
    b = true;
    for (i = 1; i <= n; i++)
      b = (b && M[i][j] == i);
    if (b)
      temp = j;
  }
  return temp;
}

int Inverse(int i)
     /* returns inverse element of i */
{
  int j = 0; 

  do {
    j++;
  } while (M[j][i] != IdentityElement);
  return j;
}

void PrintSet(Set A)
     /* prints A on a line */
{
  int i;

  for (i = 1; i<=n; i++) {
    if (A[i])
      printf("%4d ", i);
  }
  printf("\n");
}

int Cardinal(Set A)
     /* returns number of element of A */
{
  int i, compt;

  compt = 0;
  for (i = 1; i <=n; i++) {
    if (A[i])
      compt++;
  }
  return compt;
}

int CardinalOfIntersection(Set A, Set B)
     /* returns cardinal of (A intersection B) */
{
  int compt=0;
  int i;

  for (i=1; i<=n; i++)
    if (A[i])
      if (B[i])
	compt++;
  return(compt);
}

int INTER(Set A)
     /* returns INT(A) */
{
  int compt=0; int i,j,card;
  Set B;
  
  for (i=1; i<=n; i++)
    if (!(i==IdentityElement)) /* avoid set A */
      {
	SetEmptySet(B);
	for (j=1; j<=n; j++)
	  if (A[j])
	    B[M[i][j]]=true; /* B:=iA */
	card=CardinalOfIntersection(A,B);
	if (card>=compt) compt=card;
      }
  return(compt);
}

int NumberOfElementsInRightCoset(Set A, int i, int j, Partition part[kmax+1])
     /* returns the number of elements of A in the right coset number j of the partition part[i] */
{
  int compt=0; int k; 

  for (k=1; k<=n; k++)
    if (A[k]==true)
      if (part[i].split[k]==j) compt++;
  return compt;
}

boolean SatisfiesMajorationConstraintsOfEquipartitionFilter(Set A) 
     /* true iff subset A satifies majoration constraints of Equipartition filter */
{
  int i,j,d; 
  boolean b=true;

  for (i=1; ((i<=NumberOfSubgroupsForEquipartition) && (b==true)); i++)
    /* parse all subgroups involved by Equipartition filter */
    {
      d=EquipartitionPartition[i].index; /* number of right cosets of subgroup number i */
      for (j=1; ((j<=d) && (b==true)); j++)
	if (NumberOfElementsInRightCoset(A,i,j,EquipartitionPartition)>((omega/d)+1)) b=false;
    }
  return b;
}

boolean SatisfiesMinorationConstraintsOfEquipartitionFilter(Set A) 
     /* true iff subset A satisfies minoration constraints of Equipartition filter */
{
  int i,j,d; 
  boolean b=true;

  for (i=1; ((i<=NumberOfSubgroupsForEquipartition) && (b==true)); i++)
    /* parse all subgroups involved by Equipartition filter */
    {
      d=EquipartitionPartition[i].index; /* number if right cosets of subgroup number i */
      for (j=1; ((j<=d) && (b==true)); j++)
	{if (NumberOfElementsInRightCoset(A,i,j,EquipartitionPartition)<(omega/d)) b=false;}
    }
  return b;
}

int Disp(Set A, int i, Partition part[kmax+1]) 
     /* returns Disp(A) with respect to the right cosets part[i] */
{ 
  int j, compt=0;
  
  for (j=1; j<=part[i].index; j++)
    if (NumberOfElementsInRightCoset(A,i,j,part)!=0) /* if A meets the right coset number j of the partition part[i] */
      if (NumberOfElementsInRightCoset(A,i,j,part)!=(n/part[i].index)) /* if (A does not fill the right coset number j of the partition part[i]) -- (the cardinal of a right coset is equal to n/part[i].index) -- */
	compt++;
  return compt;
}

boolean SatifiesDispersionFilter(Set A) 
     /* true iff A satisfies constraints of Dispersion filter */
{ 
  int i; boolean b=true;

  for (i=1; ((i<=NumberOfSubgroups) && (b==true)); i++) 
    /* try constraints 1 and 2, which involve all subgroups */
    {
      if (Disp(A,i,SubgroupPartition)==0) b=false; /* violation of Constraint 1 */
      if (Disp(A,i,SubgroupPartition)==1) 
	if (Subgroup[i].order!=2) b=false; /* violation of Constraint 2 */
    };
  for (i=1; ((i<=NumberOfNormalSubgroups) && (b==true)); i++) 
    /* try constraint 3, which involves normal subgroups only */
    {
      if (omega!=2)
	if (Disp(A,i,NormalSubgroupPartition)==2) 
	  {
	    if ((n % 2)!=0)
	      /* case n is odd */
	      {b=false;} /* if n is odd, subgroup can not be of index 2 : violation of constraint 3 */
	    else
	      /* case n is even */
	      if (NormalSubgroup[i].order!=(n/2)) /* subgroup is not of index 2 : violation of constraint 3 (nota: to be correct, it must be ensured that n is even, as n/2 is a division in integers) */ 
		b=false;
	  };
    };
  return b;
}

void CalculateA(Set A, int last)
     /* recursive generation of subsets A: notice that A is passed by reference to keep memory usage low */
     /* last is the maximum integer of the subset A */
{int i;
 if ((CutFactor==0)||(NumberOfEntry<=CutFactor))
   for (i=last+1; i<=n; i++)
     {
       A[i]=true; /* A:=A union {i} */
       if (Cardinal(A)==omega) /* if A has reached wanted size */
	 if (A[IdentityElement]) /* if A contains IdentityElement */
	   {
	     if ((!(IntersectionFlag))||(INTER(A)<=omega-4))
	       /* if IntersectionFlag is set, keep only those A such that INTER(A)<=|A|-4, else proceed anyway */
	       {
		 /* case: Equipartition filter and Dispersion filter are not activated */
		 if ((!EquipartitionFlag) && (!DispersionFlag))
		   {
		     NumberOfEntry++;
		     PrintSet(A);
		   }
		 /* case: Dispersion filter only is activated */
		 if ((!EquipartitionFlag) && (DispersionFlag))
		   if (SatifiesDispersionFilter(A)) 
		     {
		       NumberOfEntry++;
		       PrintSet(A);
		     }
		 /* case: Equipartition filter only is activated */
		 if ((EquipartitionFlag) && (!DispersionFlag))
		   if ((SatisfiesMajorationConstraintsOfEquipartitionFilter(A)) && SatisfiesMinorationConstraintsOfEquipartitionFilter(A)) 
		     {
		       NumberOfEntry++;
		       PrintSet(A);
		     }
		 /* case: both are activated */
		 if ((EquipartitionFlag) && (DispersionFlag))
		   if (SatisfiesMajorationConstraintsOfEquipartitionFilter(A))
		     if (SatisfiesMinorationConstraintsOfEquipartitionFilter(A))
		       if (SatifiesDispersionFilter(A))
			 {
			   NumberOfEntry++;
			   PrintSet(A);
			 }
	       };
	   }
     /* make recursive calls */
     if (Cardinal(A) < omega)
       {
	 /* case: Equipartition filter is activated */
	 if (EquipartitionFlag) 
	   {
	     /* see if majoration constraints are not violated */
	     if (SatisfiesMajorationConstraintsOfEquipartitionFilter(A)) CalculateA(A,i);
	   }
	 /* case: Equipartition filter is not activated */
	 else CalculateA(A,i);
       };
     A[i]=false; /* A:=A\{i} */
   }
}

void CalculateSymmetricA(Set A, int last)
     /* recursive generation of symmetric subsets A: notice that A is passed by reference to keep memory usage low */
     /* last is the maximum integer in A */
{
  int i; 
  if ((CutFactor==0)||(NumberOfEntry<=CutFactor))
    for (i = last + 1; i <=n; i++) {
      if ((!A[i]) & (i <= inv[i])) 
	{
	  A[i] = true; 
	  A[inv[i]] = true; /* A:=A union {i,i^-1} */
	  if (Cardinal(A)==omega) /* if A has reached wanted size */
	    {
	      if ((!(IntersectionFlag))||(INTER(A)<=omega-4))
		/* if IntersectionFlag is set, keep only those A such that INTER(A)<=|A|-4, else proceeds anyway */
		{
		  /* case: Equipartition filter and Dispersion filter are not activated */
		  if ((!EquipartitionFlag) && (!DispersionFlag))
		    {
		      NumberOfEntry++;
		      PrintSet(A);
		    }
		  /* case: Dispersion filter only is activated */
		  if ((!EquipartitionFlag) && (DispersionFlag))
		    if (SatifiesDispersionFilter(A)) 
		      {
			NumberOfEntry++;
			PrintSet(A);
		      }
		  /* case: Equipartition filter only is activated */
		  if ((EquipartitionFlag) && (!DispersionFlag))
		    if ((SatisfiesMajorationConstraintsOfEquipartitionFilter(A)) && SatisfiesMinorationConstraintsOfEquipartitionFilter(A)) 
		      {
			NumberOfEntry++;
			PrintSet(A);
		      }
		  /* case: both are activated */
		  if ((EquipartitionFlag) && (DispersionFlag))
		    if (SatisfiesMajorationConstraintsOfEquipartitionFilter(A))
		      if (SatisfiesMinorationConstraintsOfEquipartitionFilter(A))
			if (SatifiesDispersionFilter(A))
			  {
			    NumberOfEntry++;
			    PrintSet(A);
			  }
		};
	    }
	/* make recursive calls */
	if (Cardinal(A) < omega)
	  {
	    /* case: Equipartition filter is activated */
	    if (EquipartitionFlag) 
	      {
		/* see if majoration constraints are not violated */
		if (SatisfiesMajorationConstraintsOfEquipartitionFilter(A)) CalculateSymmetricA(A,i);
	      }
	    /* case: Equipartition filter is not activated */
	    else CalculateSymmetricA(A,i);
	  };
	A[i] = false;
	A[inv[i]] = false; /* A:=A/{i,i^-1} */
      }
  }
}  

int main(void)
{
  FILE *error;

  /* load datas */
  scanf("%d%*[^\n]",&data); /* read RandomizedFlag */
  RandomizedFlag=(data==1);
  scanf("%d%*[^\n]",&CutFactor); /* read CutFactor */
  if (RandomizedFlag==false)
    {
      CutFactor=0; /* no reduction */
    }
  scanf("%d%*[^\n]",&data); /* read ForceSymmetricFlag */
  ForceSymmetricFlag=(data==1);
  scanf("%d%*[^\n]",&data); /* read EquipartitionFlag */
  EquipartitionFlag=(data==1);
  scanf("%d%*[^\n]",&data); /* read DispersionFlag */
  DispersionFlag=(data==1);
  scanf("%d%*[^\n]",&data); /* read IntersectionFlag */
  IntersectionFlag=(data==1);
  scanf("%d%*[^\n]",&n); /* read n, the order of the group */
  if (n>=nbmax) /* n is too big */
    {
      printf("The order of the group (%d) exceeds the maximum value allowed (%d)!!! Exiting",n,nbmax); 
      return(1);
    }
  scanf("%d%*[^\n]",&omega); /* read omega, the size of A*/
  LoadMatrix(M); /* load Cayley table of group */
  IdentityElement=Identity(M,n); 
  for (i=1; i<=n; i++) inv[i]=Inverse(i); /* initialization of inverting map */

  /* if EquipartitionFlag is set, load necessary subgroups and calculate right cosets */
  if (EquipartitionFlag==true)
    {   
      scanf("%d%*[^\n]",&NumberOfSubgroupsForEquipartition); /* read number of subgroups to be loaded */
      if (NumberOfSubgroupsForEquipartition>=kmax) /* number of subgroups is too big */ 
	{
	  error=fopen("./tmp/error","w");
	  fprintf(error,"Number of subgroups to be loaded exceeds maximum value (%d)\n",kmax);
	  fprintf(error,"Loading only the first %d subgroups \n",kmax-1);
	  fclose(error);
	  NumberOfSubgroupsForEquipartition=kmax-1;
	}
      /* load subgroups */
      for (i=1; i<=NumberOfSubgroupsForEquipartition; i++)
	{
	  scanf("%d%*[^\n]",&SubgroupForEquipartition[i].order);
	  for (j=1; j<=n; j++) SubgroupForEquipartition[i].elements[j]=false; 
	  for (j=1; j<=SubgroupForEquipartition[i].order; j++) 
	    {scanf("%d",&data); SubgroupForEquipartition[i].elements[data]=true;};
	  scanf("%*[^\n]");
	}
      /* calculate right cosets of these subgroups */
      for (i=1; i<=NumberOfSubgroupsForEquipartition; i++)
	for (j=1; j<=n; j++) EquipartitionPartition[i].split[j]=0; /* initialization */
      for (i=1; i<=NumberOfSubgroupsForEquipartition; i++)
	{
	  number=1;
	  for (j=1; j<=n; j++)
	    {
	      if (EquipartitionPartition[i].split[j]==0)
		{
		  for (k=1; k<=n; k++)
		    if (SubgroupForEquipartition[i].elements[k]==true) EquipartitionPartition[i].split[M[j][k]]=number;
		  number++;
		};
	    };
	  EquipartitionPartition[i].index=(number-1);
	};
    };
  /* if DispersionFlag is set, load necessary subgroups and calculate right cosets */
  if (DispersionFlag==true)
    {
      scanf("%d%*[^\n]",&NumberOfSubgroups); /* read number of subgroups to be loaded */
      if (NumberOfSubgroups>=kmax) /* number of subgroups is too big */ 
	{
	  error=fopen("./tmp/error","w");
	  fprintf(error,"Number of subgroups to be loaded exceeds maximum value (%d)\n",kmax);
	  fprintf(error,"Loading only the first %d subgroups \n",kmax-1);
	  fclose(error);
	  NumberOfSubgroups=kmax-1;
	}
      /* load subgroups */
      for (i=1; i<=NumberOfSubgroups; i++)
	{
	  scanf("%d%*[^\n]",&Subgroup[i].order);
	  for (j=1; j<=n; j++) Subgroup[i].elements[j]=false; 
	  for (j=1; j<=Subgroup[i].order; j++) 
	    {scanf("%d",&data); Subgroup[i].elements[data]=true;};
	  scanf("%*[^\n]");
	}
      /* calculate right cosets of these subgroups */
      for (i=1; i<=NumberOfSubgroups; i++)
	for (j=1; j<=n; j++) SubgroupPartition[i].split[j]=0; /* initialization */
      for (i=1; i<=NumberOfSubgroups; i++)
	{
	  number=1;
	  for (j=1; j<=n; j++)
	    {
	      if (SubgroupPartition[i].split[j]==0)
		{
		  for (k=1; k<=n; k++)
		    if (Subgroup[i].elements[k]==true) SubgroupPartition[i].split[M[j][k]]=number;
		  number++;
		};
	    };
	  SubgroupPartition[i].index=(number-1);
	};
      
      scanf("%d%*[^\n]",&NumberOfNormalSubgroups); /* read number of normal subgroups to be loaded */
      if (NumberOfNormalSubgroups>=kmax) /* number of subgroups is too big */ 
	{
	  error=fopen("./tmp/error","w");
	  fprintf(error,"Number of subgroups to be loaded exceeds maximum value (%d)\n",kmax);
	  fprintf(error,"Loading only the first %d subgroups \n",kmax-1);
	  fclose(error);
	  NumberOfNormalSubgroups=kmax-1;
	}
      /* load subgroups */
      for (i=1; i<=NumberOfNormalSubgroups; i++)
	{
	  scanf("%d%*[^\n]",&NormalSubgroup[i].order);
	  for (j=1; j<=n; j++) NormalSubgroup[i].elements[j]=false; 
	  for (j=1; j<=NormalSubgroup[i].order; j++) 
	    {scanf("%d",&data); NormalSubgroup[i].elements[data]=true;};
	  scanf("%*[^\n]");
	}
      /* calculate right cosets of these subgroups */
      for (i=1; i<=NumberOfNormalSubgroups; i++)
	for (j=1; j<=n; j++) NormalSubgroupPartition[i].split[j]=0; /* initialization */
      for (i=1; i<=NumberOfNormalSubgroups; i++)
	{
	  number=1;
	  for (j=1; j<=n; j++)
	    {
	      if (NormalSubgroupPartition[i].split[j]==0)
		{
		  for (k=1; k<=n; k++)
		    if (NormalSubgroup[i].elements[k]==true) NormalSubgroupPartition[i].split[M[j][k]]=number;
		  number++;
		};
	    };
	  NormalSubgroupPartition[i].index=number;
	};
    }

  /* initializations */
  SetEmptySet(A);
  if (ForceSymmetricFlag) { CalculateSymmetricA(A,0); }
					    else
					    CalculateA(A,0);
  /* write stop line */
  for (i = 1; i<=omega; i++)
    printf("%4d ", 0);
  printf("\n");
  return 0;
}
