/*-------------------------------------------------------------------
 * graph.h
 * Interface of the module graph.cc
 * Declaration of the Graph class 
 ------------------------------------------------------------------*/

#ifndef GRAPH_H
#define GRAPH_H

#include <assert.h>
#include <stdio.h>

#ifndef BOOL_DEFINED
#define BOOL_DEFINED
#define bool int
#define false 0
#define true 1
#endif

class Graph;

typedef unsigned char byte;

typedef unsigned short node_id;
enum { NULL_NODE=0xFFFF };

typedef void *param_type;

typedef void (*edge_visitor)(Graph *g, node_id n1, node_id n2,
                             param_type param);

/*----------------------------------------------------------------
 * class Graph
 * implements a (non mutable) directed graph using the adjacency
 * matrix representation 
 ---------------------------------------------------------------*/
class Graph
  { private:
      int n;              /* number of nodes  */
      byte **matrix;      /* adjacency matrix */
      int  *in_count;     /* number of 'in' edges for each node */
      node_id **in;       /* nodes connected by 'in' edges to each node */
      int  *out_count;    /* number of 'out edges for each node */
      node_id **out;      /* nodes connected by 'out' edges to each node */
    public:
      Graph(char *filename);
      Graph(FILE *f);
      ~Graph();
      int NodeCount() { return n; };
      bool HasEdge(node_id n1, node_id n2)
                      { assert(n1<n);
                        assert(n2<n);
                        return matrix[n1][n2]!=0;
                      };
      int InEdgeCount(node_id node) { assert(node<n); 
                                      return in_count[node]; 
                                    };
      int OutEdgeCount(node_id node) { assert(node<n); 
                                       return out_count[node]; 
                                     };
      int EdgeCount(node_id node) { assert(node<n); 
                                    return in_count[node]+out_count[node]; 
                                  };
      node_id GetInEdge(node_id node, unsigned short i)
                                  { assert(node<n);
                                    assert(i<in_count[node]);
                                    return in[node][i];
                                  };
      node_id GetOutEdge(node_id node, unsigned short i)
                                  { assert(node<n);
                                    assert(i<out_count[node]);
                                    return out[node][i];
                                  };
      void VisitInEdges(node_id node, edge_visitor vis, param_type param);
      void VisitOutEdges(node_id node, edge_visitor vis, param_type param);
      void VisitEdges(node_id node, edge_visitor vis, param_type param);
  };


#endif
